//===================================================================
//   Linux Kernel Ver2.4 / Ver2.6
//-------------------------------------------------------------------
//  I7sIOCTL.c
//===================================================================
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/config.h>
#include <linux/ioport.h>
#include <asm/uaccess.h>

MODULE_LICENSE( "GPL" );

#include "I7sFunc.h"

//-------------------------------------------------------
WORD   IB7_BASEADDR[MAX_IB7_NUM] = {0,0,0,0};
WORD   IB7_BASELCR [MAX_IB7_NUM] = {0,0,0,0};

IB7_DEV       Ib7Device[MAX_IB7_NUM];
static int    Ib7DeviceCounter  = 0;
static int    Ib7DeviceMask     = 0;

static struct file_operations ib7_fops;

//--------------------------------- Prototype -----------
         int  Mib7_IOCTL(struct inode*, struct file*, unsigned int, unsigned long);
         int  init_module(void);
         void cleanup_module(void);
         int  Mib7_Open (struct inode*, struct file*);
//static   void Mib7_intr_handler(int,void*,struct pt_regs*);
static   int  Mib7_Close(struct inode*, struct file*);
         int  Mib7_Initialize(void);
static   void Ib7_Pic21Check(void);
//static   int  Mib7_intr_routine(int,IB7_DEV*,struct pt_regs*);
//-------------------------------------------------------
BOOL    Initial_Ib7(WORD);
//-------------------------------------------------------

//===================================================================
//    APIC21
//===================================================================
inline  BYTE    Apic_GetLCR_IFR1(WORD i)
{
    return  _in_byte(IB7_BASELCR[i]+APIC_LCR_IFR1);
}
inline  void    Apic_SetLCR_IFR1(WORD i, BYTE data)
{
    _out_byte(IB7_BASELCR[i]+APIC_LCR_IFR1, data);
}
inline  BYTE    Apic_GetLCR_ICR1(WORD i)
{
    return  _in_byte(IB7_BASELCR[i]+APIC_LCR_ICR1);
}
inline  void    Apic_SetLCR_ICR1(WORD i, BYTE data)
{
    _out_byte(IB7_BASELCR[i]+APIC_LCR_ICR1, data);
}

//===================================================================
//(INSMOD)[1]   Mib7_IOCTL
//===================================================================
int Mib7_IOCTL(struct inode* inode, struct file* filp, unsigned int cmd, unsigned long arg)
{
  IB7_DEV      *Ib7device;
  IB7_DEV_INFO *Ib7device_info;
  int           ib7No;
  IB7_ADDACH   *AdDa;
  IB7_ENCCH    *EncCh;
  IB7_PIOCH    *PioCh;

    Ib7device = filp->private_data;
    ib7No     = MINOR(inode->i_rdev);

    switch(cmd){

          //----------------------------------------------------------
          // IB7 Reset
          //----------------------------------------------------------
     case IOCTL_IB7_RESET:
          Pio_ModeSet (ib7No);                // PIO Initialize
          Initial_Ib7(ib7No);
          break;

          //----------------------------------------------------------
          // IB7 Information
          //----------------------------------------------------------
     case IOCTL_IB7_GET_INFO:
          Ib7device_info         = (IB7_DEV_INFO*)arg;
          Ib7device_info->ib7no  = Ib7device->ib7no;
          Ib7device_info->ioLCR  = Ib7device->ioLCR;
          Ib7device_info->ioBase = Ib7device->ioBase;
          Ib7device_info->irq    = Ib7device->irq;
          Ib7device_info->Ib7ID  = Ib7device->SubSysID;
          Ib7device_info->Ib7Ver = Ib7device->Revision;

          copy_to_user((PIB7_DEV_INFO)arg,
                        Ib7device_info,
                        sizeof(IB7_DEV_INFO));
          break;

          //----------------------------------------------------------
          // IB7 A/D Read
          //----------------------------------------------------------
     case IOCTL_IB7_AD_READ:
          AdDa = (IB7_ADDACH*)arg;
          copy_from_user((DWORD*)arg, AdDa, sizeof(IB7_ADDACH));
          AdDa->wData[0] = Ad_Read(ib7No, AdDa->ch);
          copy_to_user(AdDa, (DWORD*)arg, sizeof(IB7_ADDACH));
          break;

          //----------------------------------------------------------
          // IB7 D/A Write
          //----------------------------------------------------------
     case IOCTL_IB7_DA_WRITE:
          AdDa = (IB7_ADDACH*)arg;
          copy_from_user((DWORD*)arg, AdDa, sizeof(IB7_ADDACH));
          Da_Write(ib7No, AdDa->ch, AdDa->wData[0]);
          break;

          //----------------------------------------------------------
          // IB7 ENC Mode Set
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_MODE:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          Enc_SetCounterMode(ib7No, EncCh->ch, (WORD)EncCh->mode);
          break;

          //----------------------------------------------------------
          // IB7 ENC Start
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_START:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          Enc_CounterEnable(ib7No, EncCh->ch, TRUE);
          break;

          //----------------------------------------------------------
          // IB7 ENC Read
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_READ:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          if (EncCh->ch>IB7_MAXENCCH)    return FALSE;
          EncCh->dwData[0] = Enc_ReadCounter(ib7No, EncCh->ch);
          copy_to_user(EncCh, (DWORD*)arg, sizeof(IB7_ENCCH));
          break;

          //----------------------------------------------------------
          // IB7 ENC Write
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_WRITE:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          Enc_WriteCounter(ib7No, EncCh->ch, EncCh->dwData[0]);
          break;

          //----------------------------------------------------------
          // IB7 ENC Stop
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_STOP:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          Enc_CounterEnable(ib7No, EncCh->ch, FALSE);
          break;

          //----------------------------------------------------------
          // IB7 ENC Clear
          //----------------------------------------------------------
     case IOCTL_IB7_ENC_CLEAR:
          EncCh = (IB7_ENCCH*)arg;
          copy_from_user((DWORD*)arg, EncCh, sizeof(IB7_ENCCH));
          Enc_WriteCounter(ib7No, EncCh->ch, 0);
          break;

          //----------------------------------------------------------
          // IB7 PIO Write
          //----------------------------------------------------------
     case IOCTL_IB7_PIO_WRITE:
          PioCh = (IB7_PIOCH*)arg;
          copy_from_user((DWORD*)arg, PioCh, sizeof(IB7_PIOCH));
          Pio_WriteGrp(ib7No, PioCh->OutGroup, (WORD)PioCh->wData);
          break;

          //----------------------------------------------------------
          // IB7 PIO Read
          //----------------------------------------------------------
     case IOCTL_IB7_PIO_READ:
          PioCh = (IB7_PIOCH*)arg;
          copy_from_user((DWORD*)arg, PioCh, sizeof(IB7_PIOCH));
          PioCh->wData = (DWORD)Pio_ReadGrp(ib7No, PioCh->InGroup);
          copy_to_user(PioCh, (DWORD*)arg, sizeof(IB7_PIOCH));
          break;

          //----------------------------------------------------------
          // IB7 PIO Set
          //----------------------------------------------------------
     case IOCTL_IB7_PIO_SET:
          PioCh = (IB7_PIOCH*)arg;
          copy_from_user((DWORD*)arg, PioCh, sizeof(IB7_PIOCH));
          Pio_BitSet(ib7No, PioCh->OutGroup, PioCh->ch, 1 );
          break;

          //----------------------------------------------------------
          // IB7 PIO Reset
          //----------------------------------------------------------
     case IOCTL_IB7_PIO_RESET:
          PioCh = (IB7_PIOCH*)arg;
          copy_from_user((DWORD*)arg, PioCh, sizeof(IB7_PIOCH));
          Pio_BitSet(ib7No, PioCh->OutGroup, PioCh->ch, 0 );
          break;


          //----------------------------------------------------------
          // IB7 Other
          //----------------------------------------------------------
    default:
           return -EINVAL;
    }
    return 0;
}



//===================================================================
//    Initialize IB7
//===================================================================
BOOL  Initial_Ib7(WORD Ib7No)
{
    int ch;

    for(ch=0;ch<IB7_MAXDACH;ch++)
        Da_Write(Ib7No, ch, 2048);
    Pio_WriteGrp(Ib7No, -1, 0xFFFF);
    return TRUE;
}

//===================================================================
//(INSMOD) --> init_module
//===================================================================
int init_module(void)
{
 int iRet;

    memset(&ib7_fops, 0, sizeof(struct file_operations));
    ib7_fops.ioctl   = Mib7_IOCTL;
    ib7_fops.open    = Mib7_Open;
    ib7_fops.release = Mib7_Close;

    if ( (iRet=Mib7_Initialize())!=0 )    return iRet;
    
    iRet = register_chrdev(IB7_MAJOR, IB7_DEVNAME, &ib7_fops);
    if (iRet < 0) {
        printk("<>ERR:ib7: init_module:register_chardev iRet=%d\n", iRet);
        return iRet;
    }
    printk("<>ib7 register chrdev OK.  iRet=%d\n", iRet);
    return 0;
}


//===================================================================
//  RMMOD --> cleanup_module
//===================================================================
void cleanup_module(void)
{
 int ib7No;

    for (ib7No=0; ib7No<MAX_IB7_NUM; ib7No++) {
        if (Ib7Device[ib7No].ioLCR) {
            printk("<>release LCR region, Ib7=%d, io=%08X (+%04X)\n",
                    ib7No,
                    Ib7Device[ib7No].ioLCR,
                    IB7_LCR_SIZE );
            release_region(Ib7Device[ib7No].ioLCR, IB7_LCR_SIZE);
        }
        if (Ib7Device[ib7No].ioBase) {
            printk("<>release IO region, Ib7=%d, io=%08X (+%04X)\n",
                    ib7No,
                    Ib7Device[ib7No].ioBase,
                    IB7_IO_SIZE  );
            release_region(Ib7Device[ib7No].ioBase, IB7_IO_SIZE);
        }
    }

    unregister_chrdev(IB7_MAJOR, IB7_DEVNAME);       // Release Driver
    printk("<>module removed.\n");
}




//===================================================================
//(INSMOD)  Mib7_Open
//===================================================================
int Mib7_Open(struct inode* inode, struct file* filp)
{
 int Minor;
 //int iRet;

    Minor = MINOR(inode->i_rdev);
/*    if ((iRet = request_irq(Ib7Device[Minor].irq,
                            Mib7_intr_handler,
                            SA_INTERRUPT|SA_SHIRQ,
                            IB7_DEVNAME,
                           &Ib7Device[Minor] ) ) )   return iRet;
*/
    filp->private_data=&Ib7Device[Minor];
    return 0;
}
/*
//===================================================================
//(INSMOD)  Mib7_intr_handler
//===================================================================
static void Mib7_intr_handler(int IRQ, void* pIb7Device, struct pt_regs* regs)
{
 IB7_DEV  *Ib7Device;
 int      Minor;
 BYTE     IFR, ICR;       // Inter. Flag&Clear Register
 int      iChk;

    Ib7Device = pIb7Device;
    Minor     = Ib7Device->ib7no;

    if (IRQ!=Ib7Device->irq)   return;
    
    ICR =
    IFR = Apic_GetLCR_IFR1(Minor)&0x0F;
    if (!IFR)   return;

    for (iChk=0; (iChk<MAX_APIC21_EXT_INT) && IFR; iChk++, IFR>>=1) {
       if ( !(IFR&0x01) )   continue;
       Mib7_intr_routine(iChk, pIb7Device, regs);
    }
    Apic_SetLCR_ICR1(Minor, ICR);

}
*/

//===================================================================
//(INSMOD)   Mib7_Close
//===================================================================
static int Mib7_Close(struct inode* inode, struct file* filp)
{
 IB7_DEV  *Ib7device;

    Ib7device = filp->private_data;
    Initial_Ib7(Ib7Device->ib7no);
//    free_irq(Ib7device->irq, &Ib7device);
    return 0;
}


//===================================================================
//(INSMOD)  Mib7_Initialize
//===================================================================
int Mib7_Initialize(void)
{
 int ib7No;

    Ib7_Pic21Check();
    if (Ib7DeviceCounter<=0)  return ENODEV;

    for (ib7No=0; ib7No<MAX_IB7_NUM; ib7No++) {
        if ((Ib7DeviceMask&(1<<ib7No))==0)  continue;

        IB7_BASEADDR[ib7No] = Ib7Device[ib7No].ioBase;
        IB7_BASELCR [ib7No] = Ib7Device[ib7No].ioLCR;
        Pio_ModeSet (ib7No);                // PIO Initialize
        Initial_Ib7(ib7No);
    }

    
    return 0;
}


//===================================================================
//(INSMOD)  Ib7_Pic21Check
//===================================================================
static void  Ib7_Pic21Check(void)
{
 DWORD            iBaseIo, iBaseLcr;
 BYTE             iIRQ;
 WORD             workV;
 int              ib7No , iRet;
 struct pci_dev  *ib7Device;

    Ib7DeviceCounter  = 0;
    Ib7DeviceMask     = 0;
    ib7No             = 0;

    ib7Device = NULL;
    while (ib7No<MAX_IB7_NUM) {    //  VendorID  DeviceID

        ib7Device = pci_find_device( IB7_VENDORID, IB7_DEVICEID, ib7Device );
        if (ib7Device==NULL)   break;

        iRet  = pci_read_config_word( ib7Device, PCI_SUBSYSTEM_VENDOR_ID, &workV );
        if ( (iRet < 0) || (workV != IB7_SUBVENDORID) )        continue;

        iRet = pci_read_config_word( ib7Device, PCI_SUBSYSTEM_ID, &workV );
        if ( (iRet<0) || (workV==0) )     continue;

        printk("<>find a board, Ib7no=%i, Ib7id=%i\n", ib7No, (int)workV);

        // Apic LCR Address
        iRet      = pci_read_config_dword( ib7Device, PCI_BASE_ADDRESS_0, &iBaseLcr );
        iBaseLcr &= PCI_BASE_ADDRESS_IO_MASK;

        if( NULL == request_region( iBaseLcr, IB7_LCR_SIZE, IB7_DEVNAME ) ) {
            printk("<>check Base LCR fail.\n");
            continue;
        }
        
        // I/O Base Address
        iRet     = pci_read_config_dword( ib7Device, PCI_BASE_ADDRESS_1, &iBaseIo );
        iBaseIo &= PCI_BASE_ADDRESS_IO_MASK;

        if( NULL == request_region( iBaseIo, IB7_IO_SIZE, IB7_DEVNAME ) ) {
            printk("<>check Base IO fail.\n");
            release_region( iBaseLcr, IB7_LCR_SIZE );
            continue;
        }

        // IRQ Number
        iRet = pci_read_config_byte( ib7Device, PCI_INTERRUPT_LINE, &iIRQ );
        if ( (iIRQ<MIN_IB7_IRQ) || (iIRQ>MAX_IB7_IRQ) )   continue;

        Ib7Device[ib7No].ib7no     = ib7No;
        Ib7Device[ib7No].ioLCR     = iBaseLcr;
        Ib7Device[ib7No].ioBase    = iBaseIo;
        Ib7Device[ib7No].irq       = iIRQ;
        
        Ib7Device[ib7No].VendorID  = IB7_VENDORID;
        Ib7Device[ib7No].DeviceID  = IB7_DEVICEID;
        Ib7Device[ib7No].SubSysID  = (int)workV;
        Ib7Device[ib7No].SubVenID  = IB7_VENDORID;
        Ib7Device[ib7No].Revision  = IB7_HARDWARE_VERSION;
        Ib7Device[ib7No].dwIb7DevMask = 0;
        iRet = pci_read_config_word( ib7Device,
                                    PCI_REVISION_ID,
                                    &Ib7Device[ib7No].Revision );

        Ib7DeviceMask |= 1<<ib7No;
        printk("<>Ib7Device[%d] IRQ=%d, LCR=0x%08x, BASE=0x%08x\n",
               ib7No,
               Ib7Device[ib7No].irq,
               Ib7Device[ib7No].ioLCR,
               Ib7Device[ib7No].ioBase  );
        Ib7DeviceCounter++;
        ib7No++;
    };

//  printk("<>Ib7DeviceCounter=%d, Ib7DeviceMask=%x\n", Ib7DeviceCounter, Ib7DeviceMask);
    for (ib7No=0; ib7No<Ib7DeviceCounter; ib7No++)
        Ib7Device[ib7No].dwIb7DevMask = Ib7DeviceMask;
}

/*
//===================================================================
//  Mib7_intr_routine
//===================================================================
static int Mib7_intr_routine
        (int         extintr,       // external int number(0-3)
         IB7_DEV     *dev,          // pointer to current dev struct
         struct pt_regs* regs)      // processor snapshot(rarely used)
{
    // this function should return as fast as possible, or some interrupt
    // will be lost.

    return 0;
}
*/
